function [bm,env,instf,delay]=gammatone(x,fs,cf,align)
% Implements a fourth-order gammatone filter
%
%   [bm,env,instf,delay]=gammatone(x,fs,cf,align)
%
%   This function takes an input vector and passes it
%   through a fourth-order gammatone filter.
%   
%   bm=gammatone(x,fs,cf) calculates the simulated basilar
%   membrane displacement bm, as estimated by the gammatone
%   filter, for input vector x, sampled at frequency fs and
%   at centre frequency cf.
% 
%   [bm,env,instf]=gammatone(...) optionally returns the
%   instantaneous envelope env and instantaneous frequency
%   instf.
% 
%   [...]=gammatone(...,align) allows phase alignment to be
%   applied. With align=false, no alignment is applied
%   (default). With align=true, fine structure and envelope
%   alignment is applied so that the impulse response peak
%   occurs at t=0.
% 
%   [bms,envs,instfs,delays]=gammatone(...) returns the
%   delay (in samples) removed by the phase alignment of the
%   gammatone filter, i.e. delays(n)=0 if align=false.
%
%   Adapted from code by Guy Brown, University of Sheffield
%   and Martin Cooke.
% 
%   See also GAMMATONEBANK.

% !---
% ==========================================================
% Last changed:     $Date: 2011-08-22 11:53:37 +0100 (Mon, 22 Aug 2011) $
% Last committed:   $Revision: 59 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

if nargin<3
    error('Not enough input arguments')
end
if nargin<4
    align = false;
end
if ~islogical(align)
    error('Phase correction parameter must be logical')
end
if numel(x)~=max(size(x))
    error('x must be a vector')
end
x = reshape(x,1,length(x));

if align
    B=1.019*2*pi*erb(cf);
    envelopecomptime = 3/B;
else
    envelopecomptime = 0;
end
phasealign=-2*pi*cf*envelopecomptime;
phasealign=mod(phasealign,2*pi);
phasealign=phasealign/(2*pi*cf);
shift=envelopecomptime;
intshift=round(shift*fs);

bw=1.019*erb(cf); % bandwidth
wcf=2*pi*cf; % radian frequency
tpt=(2*pi)/fs;
a=exp(-bw*tpt);
gain=((bw*tpt)^4)/6; % based on integral of impulse response

x = [x zeros(1,intshift)];
kT=(0:length(x)-1)/fs;

q=exp(1i.*(-wcf.*kT)).*x; % shift down to d.c.
p=filter([1 0],[1 -4*a 6*a^2 -4*a^3 a^4],q); % filter: part 1
u=filter([1 4*a 4*a^2 0],[1 0],p); % filter: part 2
bm=gain*real(exp(1i*wcf*(kT(intshift+1:end)+phasealign)).*u(intshift+1:end)); % shift up in frequency
env = gain*abs(u(intshift+1:end));
instf=real(cf+[diff(unwrap(angle(u(intshift+1:end)))) 0]./tpt);

delay = intshift;

% end of gammatone()

% ----------------------------------------------------------
% Local functions:
% ----------------------------------------------------------

% ----------------------------------------------------------
% erb: Equivalent rectangular bandwidth
% ----------------------------------------------------------
function y=erb(x)
y=24.7*(4.37e-3*x+1);

% [EOF]
