/*
 Calculate cross-correlograms with a wide range of options.
 Called from ch_xcorr.m.
 
 % ==========================================================
 % Last changed:     $Date: 2011-09-13 17:01:41 +0100 (Tue, 13 Sep 2011) $
 % Last committed:   $Revision: 71 $
 % Last changed by:  $Author: mu31ch $
 % ==========================================================
 */

#include "math.h"
#include "mex.h"

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	/* ====================== DECLARATIONS ====================== */
	
	/* Input left and right hair cell data */
	double *hc_L,*hc_R;
	
	/* Input number of frames in input */
	double *frameCount_in; 
	int frameCount;
	
	/* Input number of frequency channels in input HC data */
	double *numchans_in;
	mwSize numchans;
	
	/* Input frame length (samples) in input */
	double *frame_length_in;
	int frame_length;
	
	/* Input number of frames to overlap in CCG calculations */
	double *noverlap_in;
	int noverlap;
	
	/* Input maximum lag for cross-correlation */
	double *maxlag_in; 
	int maxlag;
	
	/* Time constant for cross-correlation calculations */
	double *tau_in,tau;
	
	/* Normalisation flag: 0 = no normalisation, any other value indicates normalisation */
	double *norm_flag_in;
	int norm_flag;
	
	/* Interaural coherence threshold */
	double *ic_t_in,ic_t;
	
	/* The IC for the cross-correlation at the current sample */
	double ic;
	
	/* CCGs calculated only from cross-correlations that have an IC >= IC_T */
	double *ccg_ic,ccg_ic_temp;
	mxArray *ccg_ic_mx[1];
	
	/* Inhibition data */
	double *inhib;
	
	/* Inhibition mode: 1 = multiplication, 2 = subtraction */
	double *inhib_mode_ID_in;
	int inhib_mode_ID;
	
	/* Running auto- and cross-correlations */
	double *a_LL,*a_RR,*a_LR;
	mxArray *a_LL_mx[1],*a_RR_mx[1],*a_LR_mx[1];
	
	/* Running auto- and cross-correlations for previous sample */
	double *a_LL_prev,*a_RR_prev,*a_LR_prev;
	mxArray *a_LL_prev_mx[1],*a_RR_prev_mx[1],*a_LR_prev_mx[1];
	
	/* Normalised cross-correlation */
	double *ccg_norm;
	mxArray *ccg_norm_mx[1];
	
	/* Running cross-correlation, calcuated either from a_LR or ccg_norm, depending on norm_flag */
	double *ccg_r;
	mxArray *ccg_r_mx[1];
	
	/* Output cross-correlograms */
	double *ccg_out;
	
	/* Output interaural coherence */
	double *ic_out;
	
	/* Counts number of samples over IC_T */
	int ic_count;
	
	/* Cross-correlation length (2*MAXLAG)+1 */
	int lengthCCG;
	
	/* Number of frames to look ahead for CCG calculation */
	int lookahead;
	
	/* Indices */
	mwIndex i; /* Frequency channel index */
	mwIndex j; /* Frame index */
	mwIndex n; /* Sample index */
	mwIndex m; /* Lag index */
	mwIndex sample; /* sample index */
	mwIndex ic_sample; /* IC sample index */
	mwIndex index_ccg_r; /* Running cross-correlation index */
	mwIndex index_ccg_r_ahead; /* Running cross-correlation index for look ahead frame */
	mwIndex index_a; /* Index to auto- and cross-correlations */
	mwIndex index_ccg; /* Index to CCG */
	mwIndex leftn,rightn; /* Left and right indices for calculating cross-correlations */
	
	/* Dimensions */
	mwSize ccg_dims[3]; /* CCG dimensions */
	mwSize ccg_r_dims[3]; /* Running cross-correlation dimensions */
	mwSize numsamples; /* Number of samples in HC data */
	
	/* ====================== INPUTS ====================== */
	
	hc_L = mxGetPr(prhs[0]); /* Left hair cell data */
	hc_R = mxGetPr(prhs[1]); /* Right hair cell data */
	frameCount_in = mxGetPr(prhs[2]); /* Frame count */
	frame_length_in = mxGetPr(prhs[3]); /* Frame length */
	noverlap_in = mxGetPr(prhs[4]); /* Number of frames to overlap in CCG calculations */
	maxlag_in = mxGetPr(prhs[5]); /* Maximum lag for cross-correlation */
	tau_in = mxGetPr(prhs[6]); /* Time constant for cross-correlation calculation */
	inhib = mxGetPr(prhs[7]);  /* Inhibition data */
	ic_t_in = mxGetPr(prhs[8]); /* Interaural coherence threshold */
	norm_flag_in = mxGetPr(prhs[9]); /* Normalisation flag: 0 = no normalisation, any other value indicates normalisation */
    inhib_mode_ID_in = mxGetPr(prhs[10]); /* Inhibition mode: 1 = multiplication, 2 = subtraction */
	numsamples = mxGetM(prhs[0]); /* Number of samples in HC data */
	numchans = mxGetN(prhs[0]); /* Number of frequency channels in input HC data */
	
	/* ====================== DERIVE VARIABLES ====================== */
	
	frameCount = *frameCount_in;
	frame_length = *frame_length_in;
	noverlap = (int)*noverlap_in;
	maxlag = *maxlag_in;
	tau = *tau_in;
	ic_t = *ic_t_in;
	norm_flag = (int)*norm_flag_in;
    inhib_mode_ID = (int)*inhib_mode_ID_in;
	
	lengthCCG = (2*maxlag)+1;
	ccg_dims[0] = (mwSize)lengthCCG;
	ccg_dims[1] = numchans;
	ccg_dims[2] = (mwSize)frameCount;
	ccg_r_dims[0] = (mwSize)lengthCCG;
	ccg_r_dims[1] = (mwSize)frame_length*noverlap;
	ccg_r_dims[2] = numchans;
	
	/* ====================== MATRICES ====================== */
	
	/* Running auto- and cross-correlations for previous sample */
	a_LL_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_RR_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_LR_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_LL_prev = mxGetPr(a_LL_prev_mx[0]);
	a_RR_prev = mxGetPr(a_RR_prev_mx[0]);
	a_LR_prev = mxGetPr(a_LR_prev_mx[0]);
	
	/* Running auto- and cross-correlations */
	a_LL_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_RR_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_LR_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,numchans,mxREAL);
	a_LL = mxGetPr(a_LL_mx[0]);
	a_RR = mxGetPr(a_RR_mx[0]);
	a_LR = mxGetPr(a_LR_mx[0]);
	
	/* Cross-correlations */
	ccg_norm_mx[0] = mxCreateNumericArray((mwSize)3,ccg_r_dims,mxDOUBLE_CLASS,mxREAL);
	ccg_r_mx[0] = mxCreateNumericArray((mwSize)3,ccg_r_dims,mxDOUBLE_CLASS,mxREAL);
	ccg_ic_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	ccg_norm = mxGetPr(ccg_norm_mx[0]);
	ccg_r = mxGetPr(ccg_r_mx[0]);
	ccg_ic = mxGetPr(ccg_ic_mx[0]);	
	
	/* Outputs */
	plhs[0] = mxCreateNumericArray(3,ccg_dims,mxDOUBLE_CLASS,mxREAL);
	plhs[1] = mxCreateDoubleMatrix(numsamples,numchans,mxREAL);
	ccg_out = mxGetPr(plhs[0]);
	ic_out = mxGetPr(plhs[1]);
	
	lookahead = 0; /* Initial value */
    
	/* ====================== CROSS-CORRELATE ====================== */
	for ( j = 0; j < frameCount; j++ ) {
		for ( i = 0; i < numchans; i++ ) {
			index_ccg = (i*lengthCCG)+(j*lengthCCG*numchans);
			sample = (i*(mwIndex)numsamples)+((j+lookahead)*(mwIndex)frame_length);
			ic_sample = (i*(mwIndex)numsamples)+(j*(mwIndex)frame_length);
			for ( m = 0; m < lengthCCG; m++ ) {/* reset ccg for T-F unit */
				ccg_ic[m] = 0.0;
			}
			/* Calculate raw cross-correlations */
			for ( n = 0; n < frame_length*(noverlap-lookahead); n++ ) {
				index_a = (i*lengthCCG);
				index_ccg_r = ((n+(lookahead*frame_length))+i*(noverlap*frame_length))*lengthCCG;
				ic = 0.0; /* reset the IC value for the sample */
				for ( m = 0; m < lengthCCG; m++ ) { /* calculate cross-correlations */
					leftn = sample+n+( (m>maxlag) ? (m-maxlag) : (0) );
                    rightn = sample+n+( (m<maxlag) ? (maxlag-m) : (0) );
					a_LR[index_a+m] = ((1.0/tau)*hc_L[leftn]*hc_R[rightn]) + ((1.0-(1.0/tau))*a_LR_prev[index_a+m]);
					a_LL[index_a+m] = ((1.0/tau)*hc_L[leftn]*hc_L[leftn]) + ((1.0-(1.0/tau))*a_LL_prev[index_a+m]);
					a_RR[index_a+m] = ((1.0/tau)*hc_R[rightn]*hc_R[rightn]) + ((1.0-(1.0/tau))*a_RR_prev[index_a+m]);
					ccg_norm[index_ccg_r+m] = a_LR[index_a+m]/sqrt(a_LL[index_a+m]*a_RR[index_a+m]);
					if ( (a_LL[index_a+m]*a_RR[index_a+m])==0.0 ) { /* prevent divide by zero */
						ccg_norm[index_ccg_r+m] = 0.0;
					}
					ic = ( ( (ic) > (ccg_norm[index_ccg_r+m]) ) ? (ic) : (ccg_norm[index_ccg_r+m]) ); /* calculate IC for sample (as max) */
					if ( norm_flag == 0 ) {
						/* Un-normalised */
						ccg_r[index_ccg_r+m] = a_LR[index_a+m];
					}
					else {
						/* Normalised */
						ccg_r[index_ccg_r+m] = ccg_norm[index_ccg_r+m];
					}
					/* Set cross-correlations for what will become the previous sample */
					a_LR_prev[index_a+m] = a_LR[index_a+m];
					a_LL_prev[index_a+m] = a_LL[index_a+m];
					a_RR_prev[index_a+m] = a_RR[index_a+m];
				}
				ic_out[sample+n] = ic; /* Write IC to output */
			}
			ic_count = 0; /* reset the count of samples with IC values over the IC threshold */
			/* Integrate cross-correlations */
			for ( n = 0; n < frame_length*noverlap; n++ ) {
				index_ccg_r = (n+i*(frame_length*noverlap))*lengthCCG;
				if ( ic_out[ic_sample+n] >= ic_t ) { /* check if IC exceeds IC threshold */
					ic_count += 1; /* count the number of samples that exceed the IC threshold */
					for ( m = 0; m < lengthCCG; m++ ) { /* inhibit (multiplication) */
                        if ( inhib_mode_ID==1 ) {
                            ccg_ic[m] += ccg_r[index_ccg_r+m]*inhib[sample+n];
                        }
                        else if ( inhib_mode_ID==2 ) { /* inhibit (subtraction) */
							ccg_ic_temp = ccg_r[index_ccg_r+m]-((1.0/tau)*inhib[sample+n]);
                            ccg_ic[m] += ( ( ccg_ic_temp>0 ) ? ccg_ic_temp : 0 );
                        }
                        else {
                            mexErrMsgTxt("Unknown inhib_mode_ID");
                        }
					}
				}
			}
			if ( ic_count == 0 ) { /* write zeros to output if no samples in frame have high enough IC */
				for ( m = 0; m < lengthCCG; m++ ) {
					ccg_out[index_ccg+m] = 0.0;
				}
			}
			else { /* average CCGs with high enough IC */
				for ( m = 0; m < lengthCCG; m++ ) {
					ccg_out[index_ccg+m] = ccg_ic[m]/(double)ic_count; /* Write CCG to output */
				}
			}
			/* move ccg_r's backwards to append for subsequent frames */
			for ( n = 0; n < frame_length*(noverlap-1); n++ ) {
				index_ccg_r = (n+i*(frame_length*noverlap))*lengthCCG;
				index_ccg_r_ahead = ((n+frame_length)+i*(frame_length*noverlap))*lengthCCG;
				for ( m = 0; m < lengthCCG; m++ ) {
					ccg_r[index_ccg_r+m] = ccg_r[index_ccg_r_ahead+m];
				}
			}
		} /* end frequency loop */
		lookahead = noverlap-1; /* Set value */
	} /* end frame loop */
	/* Destroy mx arrays */
	mxDestroyArray(*ccg_norm_mx);
	mxDestroyArray(*ccg_r_mx);
	mxDestroyArray(*ccg_ic_mx);
	mxDestroyArray(*a_LL_mx);
	mxDestroyArray(*a_RR_mx);
	mxDestroyArray(*a_LR_mx);
	mxDestroyArray(*a_LL_prev_mx);
	mxDestroyArray(*a_RR_prev_mx);
	mxDestroyArray(*a_LR_prev_mx);
	return;
}
